/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.apiculture;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import forestry.api.apiculture.BeeManager;
import forestry.api.apiculture.FlowerManager;
import forestry.api.apiculture.IBee;
import forestry.api.apiculture.IBeeHousing;
import forestry.api.apiculture.IBeeModifier;
import forestry.api.core.INbtReadable;
import forestry.api.core.INbtWritable;
import forestry.api.genetics.IFlowerProvider;
import forestry.core.network.IStreamable;
import forestry.core.network.PacketBufferForestry;
import forestry.core.utils.VectUtil;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3i;
import net.minecraft.world.World;

public class HasFlowersCache implements INbtWritable, INbtReadable, IStreamable {
	private static final String nbtKey = "hasFlowerCache";
	private static final String nbtKeyFlowers = "flowers";
	private static final String nbtKeyCooldown = "cooldown";

	private static final Random random = new Random();
	private static final int flowerCheckInterval = 128;

	private final int flowerCheckTime = random.nextInt(flowerCheckInterval);
	private List<BlockPos> flowerCoords = new ArrayList<>();
	private int cooldown = 0;

	private boolean needsSync = false;

	public boolean hasFlowers(IBee queen, IBeeHousing beeHousing) {
		IFlowerProvider flowerProvider = queen.getGenome().getFlowerProvider();
		String flowerType = flowerProvider.getFlowerType();
		World world = beeHousing.getWorldObj();

		if (!flowerCoords.isEmpty()) {
			if (world.func_82737_E() % flowerCheckInterval != flowerCheckTime) {
				return true;
			}

			IBeeModifier beeModifier = BeeManager.beeRoot.createBeeHousingModifier(beeHousing);
			BlockPos housingCoords = beeHousing.getCoordinates();

			Vec3i genomeTerritory = queen.getGenome().getTerritory();
			float housingModifier = beeModifier.getTerritoryModifier(queen.getGenome(), 1f);
			Vec3i area = VectUtil.scale(genomeTerritory, housingModifier * 3.0f);
			BlockPos min = VectUtil.scale(area, -0.5f).func_177971_a(housingCoords);
			BlockPos max = VectUtil.scale(area, 0.5f).func_177971_a(housingCoords);

			for (BlockPos flowerPos : flowerCoords) {
				if (!isFlowerValid(world, flowerPos, flowerType, min, max)) {
					cooldown = 0;
					break;
				}
			}
		}

		if (cooldown <= 0) {
			List<BlockPos> newFlowerCoords = FlowerManager.flowerRegistry.getAcceptedFlowerCoordinates(beeHousing, queen, flowerType, 5);
			cooldown = PluginApiculture.ticksPerBeeWorkCycle;
			if (!flowerCoords.equals(newFlowerCoords)) {
				flowerCoords = newFlowerCoords;
				needsSync = true;
			}
		} else {
			cooldown--;
		}

		return !flowerCoords.isEmpty();
	}

	public boolean needsSync() {
		boolean returnVal = needsSync;
		needsSync = false;
		return returnVal;
	}

	public void clear() {
		flowerCoords.clear();
		cooldown = 0;
	}


	public List<BlockPos> getFlowerCoords() {
		return Collections.unmodifiableList(flowerCoords);
	}

	private static boolean isFlowerValid(World world, BlockPos flowerCoords, String flowerType, BlockPos min, BlockPos max) {
		if (!isFlowerCoordInRange(flowerCoords, min, max)) {
			return false;
		}
		return FlowerManager.flowerRegistry.isAcceptedFlower(flowerType, world, flowerCoords);
	}

	private static boolean isFlowerCoordInRange(BlockPos flowerCoords, BlockPos min, BlockPos max) {
		return flowerCoords.func_177958_n() >= min.func_177958_n() && flowerCoords.func_177958_n() <= max.func_177958_n() && flowerCoords.func_177956_o() >= min.func_177956_o() && flowerCoords.func_177956_o() <= max.func_177956_o() && flowerCoords.func_177952_p() >= min.func_177952_p() && flowerCoords.func_177952_p() <= max.func_177952_p();
	}

	@Override
	public void readFromNBT(NBTTagCompound nbttagcompound) {
		if (!nbttagcompound.func_74764_b(nbtKey)) {
			return;
		}

		NBTTagCompound hasFlowerCacheNBT = nbttagcompound.func_74775_l(nbtKey);

		if (hasFlowerCacheNBT.func_74764_b(nbtKeyFlowers)) {
			int[] flowersList = hasFlowerCacheNBT.func_74759_k(nbtKeyFlowers);
			if (flowersList.length % 3 == 0) {
				int flowerCount = flowersList.length / 3;
				for (int i = 0; i < flowerCount; i++) {
					BlockPos flowerPos = new BlockPos(flowersList[i], flowersList[i + 1], flowersList[i + 2]);
					flowerCoords.add(flowerPos);
				}
				needsSync = true;
			}
		}

		cooldown = hasFlowerCacheNBT.func_74762_e(nbtKeyCooldown);
	}

	@Override
	public NBTTagCompound writeToNBT(NBTTagCompound nbttagcompound) {
		NBTTagCompound hasFlowerCacheNBT = new NBTTagCompound();

		if (!flowerCoords.isEmpty()) {
			int[] flowersList = new int[flowerCoords.size() * 3];
			int i = 0;
			for (BlockPos flowerPos : flowerCoords) {
				flowersList[i] = flowerPos.func_177958_n();
				flowersList[i + 1] = flowerPos.func_177956_o();
				flowersList[i + 2] = flowerPos.func_177952_p();
				i++;
			}

			hasFlowerCacheNBT.func_74783_a(nbtKeyFlowers, flowersList);
		}

		hasFlowerCacheNBT.func_74768_a(nbtKeyCooldown, cooldown);

		nbttagcompound.func_74782_a(nbtKey, hasFlowerCacheNBT);
		return nbttagcompound;
	}

	@Override
	public void writeData(PacketBufferForestry data) {
		int size = flowerCoords.size();
		data.func_150787_b(size);
		if (size > 0) {
			for (BlockPos pos : flowerCoords) {
				data.func_150787_b(pos.func_177958_n());
				data.func_150787_b(pos.func_177956_o());
				data.func_150787_b(pos.func_177952_p());
			}
		}
	}

	@Override
	public void readData(PacketBufferForestry data) throws IOException {
		flowerCoords.clear();

		int size = data.func_150792_a();
		while (size > 0) {
			BlockPos pos = new BlockPos(data.func_150792_a(), data.func_150792_a(), data.func_150792_a());
			flowerCoords.add(pos);
			size--;
		}
	}
}
