/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.gui;

import javax.annotation.Nullable;
import java.awt.Rectangle;
import java.io.IOException;
import java.util.List;

import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.RenderHelper;
import net.minecraft.client.renderer.texture.TextureManager;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.Slot;
import net.minecraft.util.ResourceLocation;

import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.IFluidTank;

import forestry.api.core.IErrorLogicSource;
import forestry.api.core.IErrorSource;
import forestry.api.gui.IGuiElement;
import forestry.api.gui.events.GuiEvent;
import forestry.api.gui.events.GuiEventDestination;
import forestry.core.config.Config;
import forestry.core.gui.elements.Window;
import forestry.core.gui.ledgers.ClimateLedger;
import forestry.core.gui.ledgers.HintLedger;
import forestry.core.gui.ledgers.LedgerManager;
import forestry.core.gui.ledgers.OwnerLedger;
import forestry.core.gui.ledgers.PowerLedger;
import forestry.core.gui.widgets.TankWidget;
import forestry.core.gui.widgets.Widget;
import forestry.core.gui.widgets.WidgetManager;
import forestry.core.owner.IOwnedTile;
import forestry.core.render.ColourProperties;
import forestry.core.render.ForestryResource;
import forestry.core.tiles.IClimatised;
import forestry.energy.EnergyManager;

public abstract class GuiForestry<C extends Container> extends GuiContainer implements IGuiSizable {
	protected final C container;

	public final ResourceLocation textureFile;
	protected final WidgetManager widgetManager;
	protected final LedgerManager ledgerManager;
	protected final TextLayoutHelper textLayout;
	protected final Window window;

	protected GuiForestry(String texture, C container) {
		this(new ForestryResource(texture), container);
	}

	protected GuiForestry(ResourceLocation texture, C container) {
		super(container);

		this.widgetManager = new WidgetManager(this);
		this.ledgerManager = new LedgerManager(this);
		this.window = new Window<>(field_146999_f, field_147000_g, this);

		this.textureFile = texture;

		this.container = container;

		this.textLayout = new TextLayoutHelper(this, ColourProperties.INSTANCE);
	}

	/* LEDGERS */
	@Override
	public void func_73866_w_() {
		super.func_73866_w_();

		int maxLedgerWidth = (this.field_146294_l - this.field_146999_f) / 2;

		this.ledgerManager.setMaxWidth(maxLedgerWidth);
		this.ledgerManager.clear();

		this.window.init(field_147003_i, field_147009_r);

		addLedgers();
	}

	@Override
	public void func_146280_a(Minecraft mc, int width, int height) {
		window.setSize(width, height);
		super.func_146280_a(mc, width, height);
	}

	@Override
	public void func_73876_c() {
		window.updateClient();
	}

	public void func_73863_a(int mouseX, int mouseY, float partialTicks) {
		window.setMousePosition(mouseX, mouseY);
		this.func_146276_q_();
		super.func_73863_a(mouseX, mouseY, partialTicks);
		this.func_191948_b(mouseX, mouseY);
	}

	protected abstract void addLedgers();

	protected final void addErrorLedger(IErrorSource errorSource) {
		ledgerManager.add(errorSource);
	}

	protected final void addErrorLedger(IErrorLogicSource errorSource) {
		ledgerManager.add(errorSource.getErrorLogic());
	}

	protected final void addClimateLedger(IClimatised climatised) {
		ledgerManager.add(new ClimateLedger(ledgerManager, climatised));
	}

	protected final void addPowerLedger(EnergyManager energyManager) {
		if (Config.enableEnergyStat) {
			ledgerManager.add(new PowerLedger(ledgerManager, energyManager));
		}
	}

	protected final void addHintLedger(String hintsKey) {
		if (Config.enableHints) {
			List<String> hints = Config.hints.get(hintsKey);
			addHintLedger(hints);
		}
	}

	protected final void addHintLedger(List<String> hints) {
		if (Config.enableHints) {
			if (!hints.isEmpty()) {
				ledgerManager.add(new HintLedger(ledgerManager, hints));
			}
		}
	}

	protected final void addOwnerLedger(IOwnedTile ownedTile) {
		ledgerManager.add(new OwnerLedger(ledgerManager, ownedTile));
	}

	@Override
	public void func_146281_b() {
		super.func_146281_b();
		ledgerManager.onGuiClosed();
	}

	public ColourProperties getFontColor() {
		return ColourProperties.INSTANCE;
	}

	public FontRenderer getFontRenderer() {
		return field_146289_q;
	}

	@Override
	protected void func_73864_a(int mouseX, int mouseY, int mouseButton) throws IOException {
		super.func_73864_a(mouseX, mouseY, mouseButton);

		// / Handle ledger clicks
		ledgerManager.handleMouseClicked(mouseX, mouseY, mouseButton);
		widgetManager.handleMouseClicked(mouseX, mouseY, mouseButton);
		IGuiElement origin = (window.getMousedOverElement() == null) ? this.window : this.window.getMousedOverElement();
		window.postEvent(new GuiEvent.DownEvent(origin, mouseX, mouseY, mouseButton), GuiEventDestination.ALL);
	}

	@Override
	protected void func_146286_b(int mouseX, int mouseY, int mouseButton) {
		if (widgetManager.handleMouseRelease(mouseX, mouseY, mouseButton)) {
			return;
		}
		IGuiElement origin = (window.getMousedOverElement() == null) ? this.window : this.window.getMousedOverElement();
		window.postEvent(new GuiEvent.UpEvent(origin, mouseX, mouseY, mouseButton), GuiEventDestination.ALL);
		super.func_146286_b(mouseX, mouseY, mouseButton);
	}

	@Override
	protected void func_73869_a(char typedChar, int keyCode) throws IOException {
		if (keyCode == 1 || (keyCode == this.field_146297_k.field_71474_y.field_151445_Q.func_151463_i() && this.window.getFocusedElement() == null)) {
			this.field_146297_k.field_71439_g.func_71053_j();
		}
		IGuiElement origin = (window.getFocusedElement() == null) ? this.window : this.window.getFocusedElement();
		window.postEvent(new GuiEvent.KeyEvent(origin, typedChar, keyCode), GuiEventDestination.ALL);
	}

	@Nullable
	public FluidStack getFluidStackAtPosition(int mouseX, int mouseY) {
		for (Widget widget : widgetManager.getWidgets()) {
			if (widget instanceof TankWidget && widget.isMouseOver(mouseX - field_147003_i, mouseY - field_147009_r)) {
				TankWidget tankWidget = (TankWidget) widget;
				IFluidTank tank = tankWidget.getTank();
				if (tank != null) {
					return tank.getFluid();
				}
			}
		}
		return null;
	}

	@Nullable
	protected Slot getSlotAtPosition(int mouseX, int mouseY) {
		for (int k = 0; k < this.field_147002_h.field_75151_b.size(); ++k) {
			Slot slot = this.field_147002_h.field_75151_b.get(k);

			if (isMouseOverSlot(slot, mouseX, mouseY)) {
				return slot;
			}
		}

		return null;
	}

	private boolean isMouseOverSlot(Slot par1Slot, int mouseX, int mouseY) {
		return func_146978_c(par1Slot.field_75223_e, par1Slot.field_75221_f, 16, 16, mouseX, mouseY);
	}

	@Override
	protected boolean func_193983_c(int mouseX, int mouseY, int guiLeft, int guiTop) {
		return !window.isMouseOver(mouseX, mouseY) && super.func_193983_c(mouseX, mouseY, guiLeft, guiTop);
	}

	@Override
	protected void func_146979_b(int mouseX, int mouseY) {
		ledgerManager.drawTooltips(mouseX, mouseY);

		InventoryPlayer playerInv = field_146297_k.field_71439_g.field_71071_by;

		if (playerInv.func_70445_o().func_190926_b()) {
			GuiUtil.drawToolTips(this, widgetManager.getWidgets(), mouseX, mouseY);
			GuiUtil.drawToolTips(this, field_146292_n, mouseX, mouseY);
			GuiUtil.drawToolTips(this, field_147002_h.field_75151_b, mouseX, mouseY);
			window.drawTooltip(mouseX, mouseY);
		}
	}

	@Override
	protected void func_146976_a(float f, int mouseX, int mouseY) {
		drawBackground();

		widgetManager.updateWidgets(mouseX - field_147003_i, mouseY - field_147009_r);

		RenderHelper.func_74520_c();
		GlStateManager.func_179140_f();
		GlStateManager.func_179091_B();
		GlStateManager.func_179131_c(1.0F, 1.0F, 1.0F, 1.0F);
		GlStateManager.func_179094_E();
		{
			GlStateManager.func_179109_b(field_147003_i, field_147009_r, 0.0F);
			drawWidgets();
		}
		GlStateManager.func_179121_F();

		GlStateManager.func_179124_c(1.0F, 1.0F, 1.0F);
		window.draw(mouseX, mouseY);

		bindTexture(textureFile);
	}

	protected void drawBackground() {
		bindTexture(textureFile);

		//int x = (width - xSize) / 2;
		//int y = (height - ySize) / 2;
		func_73729_b(field_147003_i, field_147009_r, 0, 0, field_146999_f, field_147000_g);
	}

	protected void drawWidgets() {
		ledgerManager.drawLedgers();
		widgetManager.drawWidgets();
	}

	protected void bindTexture(ResourceLocation texturePath) {
		GlStateManager.func_179131_c(1.0F, 1.0F, 1.0F, 1.0F);
		TextureManager textureManager = Minecraft.func_71410_x().func_110434_K();
		textureManager.func_110577_a(texturePath);
	}

	public void setZLevel(float level) {
		this.field_73735_i = level;
	}

	@Override
	public int getSizeX() {
		return field_146999_f;
	}

	@Override
	public int getSizeY() {
		return field_147000_g;
	}

	@Override
	public int getGuiLeft() {
		return field_147003_i;
	}

	@Override
	public int getGuiTop() {
		return field_147009_r;
	}

	@Override
	public Minecraft getMC() {
		return field_146297_k;
	}

	@Override
	public void func_73733_a(int par1, int par2, int par3, int par4, int par5, int par6) {
		super.func_73733_a(par1, par2, par3, par4, par5, par6);
	}

	public List<Rectangle> getExtraGuiAreas() {
		return ledgerManager.getLedgerAreas();
	}

	public TextLayoutHelper getTextLayout() {
		return textLayout;
	}
}
