/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.fluids;

import javax.annotation.Nullable;
import java.awt.Color;
import java.util.Random;

import forestry.api.core.IItemModelRegister;
import forestry.api.core.IModelManager;
import forestry.core.entities.ParticleColoredDripParticle;
import net.minecraft.block.Block;
import net.minecraft.block.BlockLiquid;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.particle.Particle;
import net.minecraft.init.Blocks;
import net.minecraft.init.SoundEvents;
import net.minecraft.item.Item;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.SoundCategory;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.fluids.BlockFluidClassic;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fml.client.FMLClientHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class BlockForestryFluid extends BlockFluidClassic implements IItemModelRegister {

	private final boolean flammable;
	private final int flammability;
	private final Color color;

	public BlockForestryFluid(Fluids forestryFluid) {
		this(forestryFluid, 0, false);
	}

	public BlockForestryFluid(Fluids forestryFluid, int flammability, boolean flammable) {
		this(forestryFluid.getFluid(), flammability, flammable, forestryFluid.getParticleColor());
	}

	private BlockForestryFluid(Fluid fluid, int flammability, boolean flammable, Color color) {
		super(fluid, Material.field_151586_h);

		setDensity(fluid.getDensity());

		this.flammability = flammability;
		this.flammable = flammable;

		this.color = color;
	}

	@Override
	public Fluid getFluid() {
		return FluidRegistry.getFluid(fluidName);
	}

	/**
	 * Copied from {@link BlockLiquid#randomDisplayTick(IBlockState, World, BlockPos, Random)}
	 * and modified to have colored particles.
	 */
	@Override
	@SideOnly(Side.CLIENT)
	public void func_180655_c(IBlockState stateIn, World worldIn, BlockPos pos, Random rand) {
		double d0 = pos.func_177958_n();
		double d1 = pos.func_177956_o();
		double d2 = pos.func_177952_p();

		if (this.field_149764_J == Material.field_151586_h) {
			int i = stateIn.func_177229_b(LEVEL);

			if (i > 0 && i < 8) {
				if (getFluid().getViscosity(worldIn, pos) < 5000 && rand.nextInt(64) == 0) {
					worldIn.func_184134_a(d0 + 0.5D, d1 + 0.5D, d2 + 0.5D, SoundEvents.field_187917_gq, SoundCategory.BLOCKS, rand.nextFloat() * 0.25F + 0.75F, rand.nextFloat() + 0.5F, false);
				}
			} else if (rand.nextInt(10) == 0) {
				worldIn.func_175688_a(EnumParticleTypes.SUSPENDED, d0 + rand.nextFloat(), d1 + rand.nextFloat(), d2 + rand.nextFloat(), 0.0D, 0.0D, 0.0D);
			}
		}

		if (this.field_149764_J == Material.field_151587_i && worldIn.func_180495_p(pos.func_177984_a()).func_185904_a() == Material.field_151579_a && !worldIn.func_180495_p(pos.func_177984_a()).func_185914_p()) {
			if (rand.nextInt(100) == 0) {
				double d8 = d0 + rand.nextFloat();
				double d4 = d1 + stateIn.func_185900_c(worldIn, pos).field_72337_e;
				double d6 = d2 + rand.nextFloat();
				worldIn.func_175688_a(EnumParticleTypes.LAVA, d8, d4, d6, 0.0D, 0.0D, 0.0D);
				worldIn.func_184134_a(d8, d4, d6, SoundEvents.field_187662_cZ, SoundCategory.BLOCKS, 0.2F + rand.nextFloat() * 0.2F, 0.9F + rand.nextFloat() * 0.15F, false);
			}

			if (rand.nextInt(200) == 0) {
				worldIn.func_184134_a(d0, d1, d2, SoundEvents.field_187656_cX, SoundCategory.BLOCKS, 0.2F + rand.nextFloat() * 0.2F, 0.9F + rand.nextFloat() * 0.15F, false);
			}
		}

		if (rand.nextInt(10) == 0 && worldIn.func_180495_p(pos.func_177977_b()).isSideSolid(worldIn, pos.func_177977_b(), EnumFacing.DOWN)) {
			Material material = worldIn.func_180495_p(pos.func_177979_c(2)).func_185904_a();

			if (!material.func_76230_c() && !material.func_76224_d()) {
				double px = d0 + rand.nextFloat();
				double py = d1 - 1.05D;
				double pz = d2 + rand.nextFloat();

				Particle fx = new ParticleColoredDripParticle(worldIn, px, py, pz, color.getRed() / 255f, color.getGreen() / 255f, color.getBlue() / 255f);
				FMLClientHandler.instance().getClient().field_71452_i.func_78873_a(fx);
			}
		}
	}

	@Override
	public boolean canDisplace(IBlockAccess world, BlockPos pos) {
		IBlockState blockState = world.func_180495_p(pos);
		return !blockState.func_185904_a().func_76224_d() &&
				super.canDisplace(world, pos);
	}

	@Override
	public boolean displaceIfPossible(World world, BlockPos pos) {
		IBlockState blockState = world.func_180495_p(pos);
		return !blockState.func_185904_a().func_76224_d() &&
				super.displaceIfPossible(world, pos);
	}

	@Override
	public int getFireSpreadSpeed(IBlockAccess world, BlockPos pos, EnumFacing face) {
		return flammable ? 30 : 0;
	}

	@Override
	public int getFlammability(IBlockAccess world, BlockPos pos, @Nullable EnumFacing face) {
		return flammability;
	}

	private static boolean isFlammable(IBlockAccess world, BlockPos pos) {
		IBlockState blockState = world.func_180495_p(pos);
		Block block = blockState.func_177230_c();
		return block.isFlammable(world, pos, EnumFacing.UP);
	}

	@Override
	public boolean isFlammable(IBlockAccess world, BlockPos pos, EnumFacing face) {
		return flammable;
	}

	@Override
	public boolean isFireSource(World world, BlockPos pos, EnumFacing side) {
		return flammable && flammability == 0;
	}

	public Color getColor() {
		return color;
	}

	@Override
	public Material func_149688_o(IBlockState state) {
		// Fahrenheit 451 = 505.928 Kelvin
		// The temperature at which book-paper catches fire, and burns.
		if (temperature > 505) {
			return Material.field_151587_i;
		} else {
			return super.func_149688_o(state);
		}
	}

	@SideOnly(Side.CLIENT)
	@Override
	public void registerModel(Item item, IModelManager manager) {

	}

	@Override
	public void func_180650_b(World world, BlockPos pos, IBlockState state, Random rand) {
		super.func_180650_b(world, pos, state, rand);

		int x = pos.func_177958_n();
		int y = pos.func_177956_o();
		int z = pos.func_177952_p();

		// Start fires if the fluid is lava-like
		if (func_149688_o(state) == Material.field_151587_i) {
			int rangeUp = rand.nextInt(3);

			for (int i = 0; i < rangeUp; ++i) {
				x += rand.nextInt(3) - 1;
				++y;
				z += rand.nextInt(3) - 1;
				IBlockState blockState = world.func_180495_p(new BlockPos(x, y, z));
				if (blockState.func_185904_a() == Material.field_151579_a) {
					if (isNeighborFlammable(world, x, y, z)) {
						world.func_175656_a(new BlockPos(x, y, z), Blocks.field_150480_ab.func_176223_P());
						return;
					}
				} else if (blockState.func_185904_a().func_76230_c()) {
					return;
				}
			}

			if (rangeUp == 0) {
				int startX = x;
				int startZ = z;

				for (int i = 0; i < 3; ++i) {
					x = startX + rand.nextInt(3) - 1;
					z = startZ + rand.nextInt(3) - 1;

					BlockPos posAbove = new BlockPos(pos.func_177958_n(), y + 1, z);
					if (world.func_175623_d(posAbove) && isFlammable(world, new BlockPos(x, y, z))) {
						world.func_175656_a(posAbove, Blocks.field_150480_ab.func_176223_P());
					}
				}
			}
		}

		// explode if very flammable and near fire
		int flammability = getFlammability(world, pos, null);
		if (flammability > 0) {
			// Explosion size is determined by flammability, up to size 4.
			float explosionSize = 4F * flammability / 300F;
			if (explosionSize > 1.0 && isNearFire(world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p())) {
				world.func_175656_a(pos, Blocks.field_150480_ab.func_176223_P());
				world.func_72885_a(null, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p(), explosionSize, true, true);
			}
		}
	}

	private static boolean isNeighborFlammable(World world, int x, int y, int z) {
		return isFlammable(world, new BlockPos(x - 1, y, z)) ||
				isFlammable(world, new BlockPos(x + 1, y, z)) ||
				isFlammable(world, new BlockPos(x, y, z - 1)) ||
				isFlammable(world, new BlockPos(x, y, z + 1)) ||
				isFlammable(world, new BlockPos(x, y - 1, z)) ||
				isFlammable(world, new BlockPos(x, y + 1, z));
	}

	private static boolean isNearFire(World world, int x, int y, int z) {
		AxisAlignedBB boundingBox = new AxisAlignedBB(x - 1, y - 1, z - 1, x + 1, y + 1, z + 1);
		return world.func_147470_e(boundingBox);
	}
}
