/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.lepidopterology.tiles;

import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.io.IOException;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.EntityLiving;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.genetics.IAllele;
import forestry.api.lepidopterology.ButterflyManager;
import forestry.api.lepidopterology.IButterfly;
import forestry.api.lepidopterology.IButterflyCocoon;
import forestry.api.lepidopterology.IButterflyGenome;
import forestry.core.network.IStreamable;
import forestry.core.network.PacketBufferForestry;
import forestry.core.network.packets.PacketTileStream;
import forestry.core.owner.IOwnedTile;
import forestry.core.owner.IOwnerHandler;
import forestry.core.owner.OwnerHandler;
import forestry.core.utils.ItemStackUtil;
import forestry.core.utils.Log;
import forestry.core.utils.NBTUtilForestry;
import forestry.core.utils.NetworkUtil;
import forestry.lepidopterology.genetics.Butterfly;
import forestry.lepidopterology.genetics.ButterflyDefinition;

public class TileCocoon extends TileEntity implements IStreamable, IOwnedTile, IButterflyCocoon {
	private final OwnerHandler ownerHandler = new OwnerHandler();
	private int age;
	private int maturationTime;
	private IButterfly caterpillar = ButterflyDefinition.CabbageWhite.getIndividual();
	private boolean isSolid;

	public TileCocoon() {
	}

	public TileCocoon(boolean isSolid) {
		this.isSolid = isSolid;
		if (isSolid) {
			this.age = 2;
		}
	}

	/* SAVING & LOADING */
	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);

		if (nbttagcompound.func_74764_b("Caterpillar")) {
			caterpillar = new Butterfly(nbttagcompound.func_74775_l("Caterpillar"));
		}
		ownerHandler.readFromNBT(nbttagcompound);
		age = nbttagcompound.func_74762_e("Age");
		maturationTime = nbttagcompound.func_74762_e("CATMAT");
		isSolid = nbttagcompound.func_74767_n("isSolid");
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);

		NBTTagCompound subcompound = new NBTTagCompound();
		caterpillar.writeToNBT(subcompound);
		nbttagcompound.func_74782_a("Caterpillar", subcompound);

		ownerHandler.writeToNBT(nbttagcompound);

		nbttagcompound.func_74768_a("Age", age);
		nbttagcompound.func_74768_a("CATMAT", maturationTime);
		nbttagcompound.func_74757_a("isSolid", isSolid);
		return nbttagcompound;
	}

	@Override
	public void writeData(PacketBufferForestry data) {
		IButterfly caterpillar = getCaterpillar();
		String speciesUID = caterpillar.getIdent();
		data.func_180714_a(speciesUID);
		data.writeInt(age);
	}

	@Override
	public void readData(PacketBufferForestry data) throws IOException {
		String speciesUID = data.readString();
		IButterfly caterpillar = getButterfly(speciesUID);
		setCaterpillar(caterpillar);
		age = data.readInt();
	}

	private static IButterfly getButterfly(String speciesUID) {
		IAllele[] butterflyTemplate = ButterflyManager.butterflyRoot.getTemplate(speciesUID);
		Preconditions.checkNotNull(butterflyTemplate, "Could not find butterfly template for species: %s", speciesUID);
		return ButterflyManager.butterflyRoot.templateAsIndividual(butterflyTemplate);
	}

	@Override
	public IOwnerHandler getOwnerHandler() {
		return ownerHandler;
	}

	@Override
	public boolean shouldRefresh(World world, BlockPos pos, IBlockState oldState, IBlockState newSate) {
		return !Block.func_149680_a(oldState.func_177230_c(), newSate.func_177230_c());
	}

	/* INETWORKEDENTITY */
	@Nullable
	@Override
	public SPacketUpdateTileEntity func_189518_D_() {
		return new SPacketUpdateTileEntity(this.func_174877_v(), 0, func_189517_E_());
	}

	@Override
	public NBTTagCompound func_189517_E_() {
		NBTTagCompound tag = super.func_189517_E_();
		return NBTUtilForestry.writeStreamableToNbt(this, tag);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void handleUpdateTag(NBTTagCompound tag) {
		int oldAge = age;
		super.handleUpdateTag(tag);
		NBTUtilForestry.readStreamableFromNbt(this, tag);
		if (oldAge != age) {
			field_145850_b.func_175704_b(field_174879_c, field_174879_c);
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void onDataPacket(NetworkManager net, SPacketUpdateTileEntity pkt) {
		super.onDataPacket(net, pkt);
		NBTTagCompound nbt = pkt.func_148857_g();
		handleUpdateTag(nbt);
	}

	public void onBlockTick() {
		maturationTime++;

		IButterflyGenome caterpillarGenome = caterpillar.getGenome();
		int caterpillarMatureTime = Math
				.round((float) caterpillarGenome.getLifespan() / (caterpillarGenome.getFertility() * 2));

		if (maturationTime >= caterpillarMatureTime) {
			if (age < 2) {
				age++;
				maturationTime = 0;
				IBlockState blockState = field_145850_b.func_180495_p(field_174879_c);
				field_145850_b.func_184138_a(field_174879_c, blockState, blockState, 0);
			} else if (caterpillar.canTakeFlight(field_145850_b, func_174877_v().func_177958_n(), func_174877_v().func_177956_o(), func_174877_v().func_177952_p())) {
				NonNullList<ItemStack> cocoonDrops = caterpillar.getCocoonDrop(this);
				for (ItemStack drop : cocoonDrops) {
					ItemStackUtil.dropItemStackAsEntity(drop, field_145850_b, field_174879_c);
				}
				field_145850_b.func_175698_g(func_174877_v());
				attemptButterflySpawn(field_145850_b, caterpillar, func_174877_v());
			}
		}
	}
	
	private boolean isListEmpty(NonNullList<ItemStack> cocoonDrops){
		if(cocoonDrops.isEmpty()){
			return true;
		}
		for(ItemStack stack : cocoonDrops){
			if(!stack.func_190926_b()){
				return false;
			}
		}
		return true;
	}

	private static void attemptButterflySpawn(World world, IButterfly butterfly, BlockPos pos) {
		EntityLiving entityLiving = ButterflyManager.butterflyRoot.spawnButterflyInWorld(world, butterfly.copy(),
				pos.func_177958_n(), pos.func_177956_o() + 0.1f, pos.func_177952_p());
		Log.trace("A caterpillar '%s' hatched at %s/%s/%s.", butterfly.getDisplayName(), pos.func_177958_n(), pos.func_177956_o(),
				pos.func_177952_p());
	}

	@Override
	public BlockPos getCoordinates() {
		return func_174877_v();
	}

	@Override
	public IButterfly getCaterpillar() {
		return caterpillar;
	}

	@Override
	public void setCaterpillar(IButterfly butterfly) {
		this.caterpillar = butterfly;
		sendNetworkUpdate();
	}

	private void sendNetworkUpdate() {
		NetworkUtil.sendNetworkPacket(new PacketTileStream(this), field_174879_c, field_145850_b);
	}

	public int getAge() {
		return age;
	}

	public void setAge(int age) {
		this.age = age;
	}

	public NonNullList<ItemStack> getCocoonDrops() {
		return caterpillar.getCocoonDrop(this);
	}

	@Override
	public boolean isSolid() {
		return isSolid;
	}

}
