/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.render;

import java.awt.Color;
import java.util.EnumMap;
import java.util.Locale;

import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.model.ModelBase;
import net.minecraft.client.model.ModelRenderer;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.texture.TextureManager;
import net.minecraft.client.renderer.tileentity.TileEntitySpecialRenderer;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ResourceLocation;
import net.minecraft.world.World;

import forestry.core.blocks.BlockBase;
import forestry.core.fluids.Fluids;
import forestry.core.tiles.IRenderableTile;
import forestry.core.tiles.TileBase;

public class RenderMachine extends TileEntitySpecialRenderer<TileBase> {

	private final ModelRenderer basefront;
	private final ModelRenderer baseback;
	private final ModelRenderer resourceTank;
	private final ModelRenderer productTank;

	private final ResourceLocation textureBase;
	private final ResourceLocation textureResourceTank;
	private final ResourceLocation textureProductTank;

	private final EnumMap<EnumTankLevel, ResourceLocation> texturesTankLevels = new EnumMap<>(EnumTankLevel.class);

	public RenderMachine(String baseTexture) {
		ModelBase model = new RenderModelBase();

		basefront = new ModelRenderer(model, 0, 0);
		basefront.func_78789_a(-8F, -8F, -8F, 16, 4, 16);
		basefront.field_78800_c = 8;
		basefront.field_78797_d = 8;
		basefront.field_78798_e = 8;

		baseback = new ModelRenderer(model, 0, 0);
		baseback.func_78789_a(-8F, 4F, -8F, 16, 4, 16);
		baseback.field_78800_c = 8;
		baseback.field_78797_d = 8;
		baseback.field_78798_e = 8;

		resourceTank = new ModelRenderer(model, 0, 0);
		resourceTank.func_78789_a(-6F, -8F, -6F, 12, 16, 6);
		resourceTank.field_78800_c = 8;
		resourceTank.field_78797_d = 8;
		resourceTank.field_78798_e = 8;

		productTank = new ModelRenderer(model, 0, 0);
		productTank.func_78789_a(-6F, -8F, 0F, 12, 16, 6);
		productTank.field_78800_c = 8;
		productTank.field_78797_d = 8;
		productTank.field_78798_e = 8;

		textureBase = new ForestryResource(baseTexture + "base.png");
		textureProductTank = new ForestryResource(baseTexture + "tank_product_empty.png");
		textureResourceTank = new ForestryResource(baseTexture + "tank_resource_empty.png");

		for (EnumTankLevel tankLevel : EnumTankLevel.values()) {
			if (tankLevel == EnumTankLevel.EMPTY) {
				continue;
			}
			String tankLevelString = tankLevel.toString().toLowerCase(Locale.ENGLISH);
			texturesTankLevels.put(tankLevel, new ForestryResource("textures/blocks/machine_tank_" + tankLevelString + ".png"));
		}
	}
	
	/**
	 * @param tile If it null its render the item else it render the tile entity.
	 */
	@Override
	public void func_192841_a(TileBase tile, double x, double y, double z, float partialTicks, int destroyStage, float alpha) {
		if (tile != null) {
			IRenderableTile generator = (IRenderableTile) tile;
			World worldObj = tile.getWorldObj();
			if (worldObj.func_175667_e(tile.func_174877_v())) {
				IBlockState blockState = worldObj.func_180495_p(tile.func_174877_v());
				if (blockState.func_177230_c() instanceof BlockBase) {
					EnumFacing facing = blockState.func_177229_b(BlockBase.FACING);
					render(generator.getResourceTankInfo(), generator.getProductTankInfo(), facing, x, y, z, destroyStage);
					return;
				}
			}
		}
		render(TankRenderInfo.EMPTY, TankRenderInfo.EMPTY, EnumFacing.SOUTH, x, y, z, -1);
	}

	private void render(TankRenderInfo resourceTankInfo, TankRenderInfo productTankInfo, EnumFacing orientation, double x, double y, double z, int destroyStage) {
		GlStateManager.func_179094_E();
		GlStateManager.func_179109_b((float) x, (float) y, (float) z);
		float[] angle = {0, 0, 0};

		switch (orientation) {
			case EAST:
				angle[1] = (float) Math.PI;
				angle[2] = (float) -Math.PI / 2;
				break;
			case WEST:
				angle[2] = (float) Math.PI / 2;
				break;
			case UP:
				break;
			case DOWN:
				angle[2] = (float) Math.PI;
				break;
			case SOUTH:
				angle[0] = (float) Math.PI / 2;
				angle[2] = (float) Math.PI / 2;
				break;
			case NORTH:
			default:
				angle[0] = (float) -Math.PI / 2;
				angle[2] = (float) Math.PI / 2;
				break;
		}

		basefront.field_78795_f = angle[0];
		basefront.field_78796_g = angle[1];
		basefront.field_78808_h = angle[2];

		baseback.field_78795_f = angle[0];
		baseback.field_78796_g = angle[1];
		baseback.field_78808_h = angle[2];

		resourceTank.field_78795_f = angle[0];
		resourceTank.field_78796_g = angle[1];
		resourceTank.field_78808_h = angle[2];

		productTank.field_78795_f = angle[0];
		productTank.field_78796_g = angle[1];
		productTank.field_78808_h = angle[2];

		float factor = (float) (1.0 / 16.0);

		TextureManager textureManager = Minecraft.func_71410_x().func_110434_K();
		textureManager.func_110577_a(textureBase);

		basefront.func_78785_a(factor);
		baseback.func_78785_a(factor);

		renderTank(resourceTank, textureResourceTank, resourceTankInfo, factor);
		renderTank(productTank, textureProductTank, productTankInfo, factor);

		GlStateManager.func_179121_F();
		GlStateManager.func_179131_c(1.0F, 1.0F, 1.0F, 1.0F);
	}

	private void renderTank(ModelRenderer tankModel, ResourceLocation textureBase, TankRenderInfo renderInfo, float factor) {
		TextureManager textureManager = Minecraft.func_71410_x().func_110434_K();
		textureManager.func_110577_a(textureBase);
		tankModel.func_78785_a(factor);

		ResourceLocation textureResourceTankLevel = texturesTankLevels.get(renderInfo.getLevel());
		if (textureResourceTankLevel == null) {
			return;
		}

		// TODO: render fluid overlay on tank
		Fluids fluidDefinition = Fluids.getFluidDefinition(renderInfo.getFluidStack());
		Color primaryTankColor = fluidDefinition == null ? Color.BLUE : fluidDefinition.getParticleColor();
		float[] colors = new float[3];
		primaryTankColor.getRGBColorComponents(colors);
		GlStateManager.func_179131_c(colors[0], colors[1], colors[2], 1.0f);

		textureManager.func_110577_a(textureResourceTankLevel);
		tankModel.func_78785_a(factor);

		GlStateManager.func_179131_c(1.0f, 1.0f, 1.0f, 1.0f);
	}

	private static class RenderModelBase extends ModelBase {
	}
}
