/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.inventory;

import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.util.Random;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.ItemStackHelper;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.TextComponentString;

import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.common.capabilities.ICapabilityProvider;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.wrapper.InvWrapper;

import forestry.core.tiles.IFilterSlotDelegate;

public abstract class ItemInventory implements IInventory, IFilterSlotDelegate, ICapabilityProvider {
	private static final String KEY_SLOTS = "Slots";
	private static final String KEY_UID = "UID";
	private static final Random rand = new Random();

	private final IItemHandler itemHandler = new InvWrapper(this);

	protected final EntityPlayer player;
	private final ItemStack parent;
	private final NonNullList<ItemStack> inventoryStacks;

	public ItemInventory(EntityPlayer player, int size, ItemStack parent) {
		Preconditions.checkArgument(!parent.func_190926_b(), "Parent cannot be empty.");

		this.player = player;
		this.parent = parent;
		this.inventoryStacks = NonNullList.func_191197_a(size, ItemStack.field_190927_a);

		NBTTagCompound nbt = parent.func_77978_p();
		if (nbt == null) {
			nbt = new NBTTagCompound();
			parent.func_77982_d(nbt);
		}
		setUID(nbt); // Set a uid to identify the itemStack on SMP

		NBTTagCompound nbtSlots = nbt.func_74775_l(KEY_SLOTS);
		for (int i = 0; i < inventoryStacks.size(); i++) {
			String slotKey = getSlotNBTKey(i);
			if (nbtSlots.func_74764_b(slotKey)) {
				NBTTagCompound itemNbt = nbtSlots.func_74775_l(slotKey);
				ItemStack itemStack = new ItemStack(itemNbt);
				inventoryStacks.set(i, itemStack);
			} else {
				inventoryStacks.set(i, ItemStack.field_190927_a);
			}
		}
	}

	public static int getOccupiedSlotCount(ItemStack itemStack) {
		NBTTagCompound nbt = itemStack.func_77978_p();
		if (nbt == null) {
			return 0;
		}

		NBTTagCompound slotNbt = nbt.func_74775_l(KEY_SLOTS);
		return slotNbt.func_150296_c().size();
	}

	private void setUID(NBTTagCompound nbt) {
		if (!nbt.func_74764_b(KEY_UID)) {
			nbt.func_74768_a(KEY_UID, rand.nextInt());
		}
	}

	public boolean isParentItemInventory(ItemStack itemStack) {
		ItemStack parent = getParent();
		return isSameItemInventory(parent, itemStack);
	}

	protected ItemStack getParent() {
		for (EnumHand hand : EnumHand.values()) {
			ItemStack held = player.func_184586_b(hand);
			if (isSameItemInventory(held, parent)) {
				return held;
			}
		}
		return parent;
	}

	private static boolean isSameItemInventory(ItemStack base, ItemStack comparison) {
		if (base.func_190926_b() || comparison.func_190926_b()) {
			return false;
		}

		if (base.func_77973_b() != comparison.func_77973_b()) {
			return false;
		}

		NBTTagCompound baseTagCompound = base.func_77978_p();
		NBTTagCompound comparisonTagCompound = comparison.func_77978_p();
		if (baseTagCompound == null || comparisonTagCompound == null) {
			return false;
		}

		if (!baseTagCompound.func_74764_b(KEY_UID) || !comparisonTagCompound.func_74764_b(KEY_UID)) {
			return false;
		}

		int baseUID = baseTagCompound.func_74762_e(KEY_UID);
		int comparisonUID = comparisonTagCompound.func_74762_e(KEY_UID);
		return baseUID == comparisonUID;
	}

	private void writeToParentNBT() {
		ItemStack parent = getParent();

		NBTTagCompound nbt = parent.func_77978_p();
		if (nbt == null) {
			nbt = new NBTTagCompound();
			parent.func_77982_d(nbt);
		}

		NBTTagCompound slotsNbt = new NBTTagCompound();
		for (int i = 0; i < func_70302_i_(); i++) {
			ItemStack itemStack = func_70301_a(i);
			if (!itemStack.func_190926_b()) {
				String slotKey = getSlotNBTKey(i);
				NBTTagCompound itemNbt = new NBTTagCompound();
				itemStack.func_77955_b(itemNbt);
				slotsNbt.func_74782_a(slotKey, itemNbt);
			}
		}

		nbt.func_74782_a(KEY_SLOTS, slotsNbt);
	}

	private static String getSlotNBTKey(int i) {
		return Integer.toString(i, Character.MAX_RADIX);
	}

	public void onSlotClick(int slotIndex, EntityPlayer player) {
	}

	@Override
	public boolean func_191420_l() {
		for (ItemStack itemstack : this.inventoryStacks) {
			if (!itemstack.func_190926_b()) {
				return false;
			}
		}

		return true;
	}


	@Override
	public ItemStack func_70298_a(int index, int count) {
		ItemStack itemstack = ItemStackHelper.func_188382_a(this.inventoryStacks, index, count);

		if (!itemstack.func_190926_b()) {
			this.func_70296_d();
		}

		return itemstack;
	}

	@Override
	public void func_70299_a(int index, ItemStack itemstack) {
		inventoryStacks.set(index, itemstack);

		ItemStack parent = getParent();

		NBTTagCompound nbt = parent.func_77978_p();
		if (nbt == null) {
			nbt = new NBTTagCompound();
			parent.func_77982_d(nbt);
		}

		NBTTagCompound slotNbt;
		if (!nbt.func_74764_b(KEY_SLOTS)) {
			slotNbt = new NBTTagCompound();
			nbt.func_74782_a(KEY_SLOTS, slotNbt);
		} else {
			slotNbt = nbt.func_74775_l(KEY_SLOTS);
		}

		String slotKey = getSlotNBTKey(index);

		if (itemstack.func_190926_b()) {
			slotNbt.func_82580_o(slotKey);
		} else {
			NBTTagCompound itemNbt = new NBTTagCompound();
			itemstack.func_77955_b(itemNbt);

			slotNbt.func_74782_a(slotKey, itemNbt);
		}
	}

	@Override
	public ItemStack func_70301_a(int i) {
		return inventoryStacks.get(i);
	}

	@Override
	public int func_70302_i_() {
		return inventoryStacks.size();
	}

	@Override
	public String func_70005_c_() {
		return "BeeBag";
	}

	@Override
	public ITextComponent func_145748_c_() {
		return new TextComponentString(func_70005_c_());
	}

	@Override
	public int func_70297_j_() {
		return 64;
	}

	@Override
	public final void func_70296_d() {
		writeToParentNBT();
	}

	@Override
	public boolean func_70300_a(EntityPlayer entityplayer) {
		return true;
	}

	@Override
	public boolean func_145818_k_() {
		return true;
	}

	@Override
	public boolean func_94041_b(int slotIndex, ItemStack itemStack) {
		return canSlotAccept(slotIndex, itemStack);
	}

	@Override
	public void func_174889_b(EntityPlayer player) {
	}

	@Override
	public void func_174886_c(EntityPlayer player) {
	}

	@Override
	public ItemStack func_70304_b(int slot) {
		ItemStack toReturn = func_70301_a(slot);

		if (!toReturn.func_190926_b()) {
			func_70299_a(slot, ItemStack.field_190927_a);
		}

		return toReturn;
	}

	/* IFilterSlotDelegate */
	@Override
	public boolean canSlotAccept(int slotIndex, ItemStack itemStack) {
		return true;
	}

	@Override
	public boolean isLocked(int slotIndex) {
		return false;
	}

	/* Fields */
	@Override
	public int func_174887_a_(int id) {
		return 0;
	}

	@Override
	public int func_174890_g() {
		return 0;
	}

	@Override
	public void func_174888_l() {
	}

	@Override
	public void func_174885_b(int id, int value) {
	}

	@Override
	@Nullable
	public <T> T getCapability(Capability<T> capability, @Nullable EnumFacing facing) {
		if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY) {
			return CapabilityItemHandler.ITEM_HANDLER_CAPABILITY.cast(itemHandler);
		}
		return null;
	}

	@Override
	public boolean hasCapability(Capability<?> capability, @Nullable EnumFacing facing) {
		return capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY;
	}

	public IItemHandler getItemHandler() {
		return itemHandler;
	}
}
